/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.runtime;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.clients.consumer.OffsetCommitCallback;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.data.SchemaAndValue;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.errors.RetriableException;
import org.apache.kafka.connect.runtime.WorkerConfig;
import org.apache.kafka.connect.runtime.WorkerSinkTaskContext;
import org.apache.kafka.connect.runtime.WorkerSinkTaskThread;
import org.apache.kafka.connect.runtime.WorkerTask;
import org.apache.kafka.connect.sink.SinkRecord;
import org.apache.kafka.connect.sink.SinkTask;
import org.apache.kafka.connect.sink.SinkTaskContext;
import org.apache.kafka.connect.storage.Converter;
import org.apache.kafka.connect.util.ConnectorTaskId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class WorkerSinkTask
implements WorkerTask {
    private static final Logger log = LoggerFactory.getLogger(WorkerSinkTask.class);
    private final ConnectorTaskId id;
    private final SinkTask task;
    private final WorkerConfig workerConfig;
    private final Time time;
    private final Converter keyConverter;
    private final Converter valueConverter;
    private WorkerSinkTaskThread workThread;
    private Map<String, String> taskProps;
    private KafkaConsumer<byte[], byte[]> consumer;
    private WorkerSinkTaskContext context;
    private boolean started;
    private final List<SinkRecord> messageBatch;
    private Map<TopicPartition, OffsetAndMetadata> lastCommittedOffsets;
    private Map<TopicPartition, OffsetAndMetadata> currentOffsets;
    private boolean pausedForRedelivery;

    public WorkerSinkTask(ConnectorTaskId id, SinkTask task, WorkerConfig workerConfig, Converter keyConverter, Converter valueConverter, Time time) {
        this.id = id;
        this.task = task;
        this.workerConfig = workerConfig;
        this.keyConverter = keyConverter;
        this.valueConverter = valueConverter;
        this.time = time;
        this.started = false;
        this.messageBatch = new ArrayList<SinkRecord>();
        this.currentOffsets = new HashMap<TopicPartition, OffsetAndMetadata>();
        this.pausedForRedelivery = false;
    }

    @Override
    public void start(Map<String, String> props) {
        this.taskProps = props;
        this.consumer = this.createConsumer();
        this.context = new WorkerSinkTaskContext(this.consumer);
        this.workThread = this.createWorkerThread();
        this.workThread.start();
    }

    @Override
    public void stop() {
        if (this.workThread != null) {
            this.workThread.startGracefulShutdown();
        }
        this.consumer.wakeup();
    }

    @Override
    public boolean awaitStop(long timeoutMs) {
        boolean success = true;
        if (this.workThread != null) {
            try {
                success = this.workThread.awaitShutdown(timeoutMs, TimeUnit.MILLISECONDS);
                if (!success) {
                    this.workThread.forceShutdown();
                }
            }
            catch (InterruptedException e) {
                success = false;
            }
        }
        this.task.stop();
        return success;
    }

    @Override
    public void close() {
        if (this.consumer != null) {
            this.consumer.close();
        }
    }

    public boolean joinConsumerGroupAndStart() {
        String topicsStr = this.taskProps.get("topics");
        if (topicsStr == null || topicsStr.isEmpty()) {
            throw new ConnectException("Sink tasks require a list of topics.");
        }
        String[] topics = topicsStr.split(",");
        log.debug("Task {} subscribing to topics {}", (Object)this.id, (Object)topics);
        this.consumer.subscribe(Arrays.asList(topics), (ConsumerRebalanceListener)new HandleRebalance());
        try {
            this.consumer.poll(0L);
        }
        catch (WakeupException e) {
            log.error("Sink task {} was stopped before completing join group. Task initialization and start is being skipped", (Object)this);
            return false;
        }
        this.task.initialize((SinkTaskContext)this.context);
        this.task.start(this.taskProps);
        log.info("Sink task {} finished initialization and start", (Object)this);
        this.started = true;
        return true;
    }

    public void poll(long timeoutMs) {
        try {
            this.rewind();
            long retryTimeout = this.context.timeout();
            if (retryTimeout > 0L) {
                timeoutMs = Math.min(timeoutMs, retryTimeout);
                this.context.timeout(-1L);
            }
            log.trace("{} polling consumer with timeout {} ms", (Object)this.id, (Object)timeoutMs);
            ConsumerRecords msgs = this.consumer.poll(timeoutMs);
            assert (this.messageBatch.isEmpty() || msgs.isEmpty());
            log.trace("{} polling returned {} messages", (Object)this.id, (Object)msgs.count());
            this.convertMessages((ConsumerRecords<byte[], byte[]>)msgs);
            this.deliverMessages();
        }
        catch (WakeupException we) {
            log.trace("{} consumer woken up", (Object)this.id);
        }
    }

    public void commitOffsets(boolean sync, final int seqno) {
        log.info("{} Committing offsets", (Object)this);
        HashMap<TopicPartition, OffsetAndMetadata> offsets = new HashMap<TopicPartition, OffsetAndMetadata>(this.currentOffsets);
        try {
            this.task.flush(offsets);
        }
        catch (Throwable t) {
            log.error("Commit of {} offsets failed due to exception while flushing:", (Object)this, (Object)t);
            log.error("Rewinding offsets to last committed offsets");
            for (Map.Entry<TopicPartition, OffsetAndMetadata> entry : this.lastCommittedOffsets.entrySet()) {
                log.debug("{} Rewinding topic partition {} to offset {}", new Object[]{this.id, entry.getKey(), entry.getValue().offset()});
                this.consumer.seek(entry.getKey(), entry.getValue().offset());
            }
            this.currentOffsets = new HashMap<TopicPartition, OffsetAndMetadata>(this.lastCommittedOffsets);
            this.workThread.onCommitCompleted(t, seqno);
            return;
        }
        if (sync) {
            try {
                this.consumer.commitSync(offsets);
                this.lastCommittedOffsets = offsets;
                this.workThread.onCommitCompleted(null, seqno);
            }
            catch (KafkaException e) {
                this.workThread.onCommitCompleted(e, seqno);
            }
        } else {
            OffsetCommitCallback cb = new OffsetCommitCallback(){

                public void onComplete(Map<TopicPartition, OffsetAndMetadata> offsets, Exception error) {
                    WorkerSinkTask.this.lastCommittedOffsets = offsets;
                    WorkerSinkTask.this.workThread.onCommitCompleted(error, seqno);
                }
            };
            this.consumer.commitAsync(offsets, cb);
        }
    }

    public Time time() {
        return this.time;
    }

    public WorkerConfig workerConfig() {
        return this.workerConfig;
    }

    private KafkaConsumer<byte[], byte[]> createConsumer() {
        KafkaConsumer newConsumer;
        HashMap<String, String> props = new HashMap<String, String>();
        props.put("group.id", "connect-" + this.id.connector());
        props.put("bootstrap.servers", Utils.join((Collection)this.workerConfig.getList("bootstrap.servers"), (String)","));
        props.put("enable.auto.commit", "false");
        props.put("auto.offset.reset", "earliest");
        props.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
        props.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
        props.putAll(this.workerConfig.originalsWithPrefix("consumer."));
        try {
            newConsumer = new KafkaConsumer(props);
        }
        catch (Throwable t) {
            throw new ConnectException("Failed to create consumer", t);
        }
        return newConsumer;
    }

    private WorkerSinkTaskThread createWorkerThread() {
        return new WorkerSinkTaskThread(this, "WorkerSinkTask-" + this.id, this.time, this.workerConfig);
    }

    private void convertMessages(ConsumerRecords<byte[], byte[]> msgs) {
        for (ConsumerRecord msg : msgs) {
            log.trace("Consuming message with key {}, value {}", msg.key(), msg.value());
            SchemaAndValue keyAndSchema = this.keyConverter.toConnectData(msg.topic(), (byte[])msg.key());
            SchemaAndValue valueAndSchema = this.valueConverter.toConnectData(msg.topic(), (byte[])msg.value());
            this.messageBatch.add(new SinkRecord(msg.topic(), msg.partition(), keyAndSchema.schema(), keyAndSchema.value(), valueAndSchema.schema(), valueAndSchema.value(), msg.offset()));
        }
    }

    private void deliverMessages() {
        try {
            this.task.put(new ArrayList<SinkRecord>(this.messageBatch));
            for (SinkRecord record : this.messageBatch) {
                this.currentOffsets.put(new TopicPartition(record.topic(), record.kafkaPartition().intValue()), new OffsetAndMetadata(record.kafkaOffset() + 1L));
            }
            this.messageBatch.clear();
            if (this.pausedForRedelivery) {
                for (TopicPartition tp : this.consumer.assignment()) {
                    if (this.context.pausedPartitions().contains(tp)) continue;
                    this.consumer.resume(new TopicPartition[]{tp});
                }
                this.pausedForRedelivery = false;
            }
        }
        catch (RetriableException e) {
            log.error("RetriableException from SinkTask {}:", (Object)this.id, (Object)e);
            this.pausedForRedelivery = true;
            for (TopicPartition tp : this.consumer.assignment()) {
                this.consumer.pause(new TopicPartition[]{tp});
            }
        }
        catch (Throwable t) {
            log.error("Task {} threw an uncaught and unrecoverable exception", (Object)this.id);
            log.error("Task is being killed and will not recover until manually restarted:", t);
            throw new ConnectException("Exiting WorkerSinkTask due to unrecoverable exception.");
        }
    }

    private void rewind() {
        Map<TopicPartition, Long> offsets = this.context.offsets();
        if (offsets.isEmpty()) {
            return;
        }
        for (TopicPartition tp : offsets.keySet()) {
            Long offset = offsets.get(tp);
            if (offset == null) continue;
            log.trace("Rewind {} to offset {}.", (Object)tp, (Object)offset);
            this.consumer.seek(tp, offset.longValue());
            this.lastCommittedOffsets.put(tp, new OffsetAndMetadata(offset.longValue()));
            this.currentOffsets.put(tp, new OffsetAndMetadata(offset.longValue()));
        }
        this.context.clearOffsets();
    }

    private class HandleRebalance
    implements ConsumerRebalanceListener {
        private HandleRebalance() {
        }

        public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
            WorkerSinkTask.this.lastCommittedOffsets = new HashMap();
            WorkerSinkTask.this.currentOffsets = new HashMap();
            for (TopicPartition tp : partitions) {
                long pos = WorkerSinkTask.this.consumer.position(tp);
                WorkerSinkTask.this.lastCommittedOffsets.put(tp, new OffsetAndMetadata(pos));
                WorkerSinkTask.this.currentOffsets.put(tp, new OffsetAndMetadata(pos));
                log.debug("{} assigned topic partition {} with offset {}", new Object[]{WorkerSinkTask.this.id, tp, pos});
            }
            if (WorkerSinkTask.this.pausedForRedelivery) {
                WorkerSinkTask.this.pausedForRedelivery = false;
                HashSet<TopicPartition> assigned = new HashSet<TopicPartition>(partitions);
                Set<TopicPartition> taskPaused = WorkerSinkTask.this.context.pausedPartitions();
                for (TopicPartition tp : partitions) {
                    if (taskPaused.contains(tp)) continue;
                    WorkerSinkTask.this.consumer.resume(new TopicPartition[]{tp});
                }
                Iterator<TopicPartition> tpIter = taskPaused.iterator();
                while (tpIter.hasNext()) {
                    TopicPartition tp;
                    tp = tpIter.next();
                    if (!assigned.contains(tp)) continue;
                    tpIter.remove();
                }
            }
            if (WorkerSinkTask.this.started) {
                WorkerSinkTask.this.task.onPartitionsAssigned(partitions);
            }
        }

        public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
            if (WorkerSinkTask.this.started) {
                WorkerSinkTask.this.task.onPartitionsRevoked(partitions);
                WorkerSinkTask.this.commitOffsets(true, -1);
            }
            WorkerSinkTask.this.messageBatch.clear();
        }
    }
}

