/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.availability;

import java.time.Clock;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.neo4j.helpers.Format;
import org.neo4j.helpers.Listeners;
import org.neo4j.helpers.collection.Iterables;
import org.neo4j.kernel.availability.AvailabilityGuard;
import org.neo4j.kernel.availability.AvailabilityListener;
import org.neo4j.kernel.availability.AvailabilityRequirement;
import org.neo4j.kernel.availability.UnavailableException;
import org.neo4j.logging.Log;

public class DatabaseAvailabilityGuard
implements AvailabilityGuard {
    private static final String DATABASE_AVAILABLE_MSG = "Fulfilling of requirement '%s' makes database %s available.";
    private static final String DATABASE_UNAVAILABLE_MSG = "Requirement `%s` makes database %s unavailable.";
    private final AtomicInteger requirementCount = new AtomicInteger(0);
    private final Set<AvailabilityRequirement> blockingRequirements = new CopyOnWriteArraySet<AvailabilityRequirement>();
    private final AtomicBoolean isShutdown = new AtomicBoolean(false);
    private final Listeners<AvailabilityListener> listeners = new Listeners();
    private final String databaseName;
    private final Clock clock;
    private final Log log;

    public DatabaseAvailabilityGuard(String databaseName, Clock clock, Log log) {
        this.databaseName = databaseName;
        this.clock = clock;
        this.log = log;
        this.listeners.add(new LoggingAvailabilityListener(log, databaseName));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void require(AvailabilityRequirement requirement) {
        if (!this.blockingRequirements.add(requirement)) {
            return;
        }
        AtomicInteger atomicInteger = this.requirementCount;
        synchronized (atomicInteger) {
            if (this.requirementCount.getAndIncrement() == 0 && !this.isShutdown.get()) {
                this.log.info(DATABASE_UNAVAILABLE_MSG, requirement.description(), this.databaseName);
                this.listeners.notify(AvailabilityListener::unavailable);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void fulfill(AvailabilityRequirement requirement) {
        if (!this.blockingRequirements.remove(requirement)) {
            return;
        }
        AtomicInteger atomicInteger = this.requirementCount;
        synchronized (atomicInteger) {
            if (this.requirementCount.getAndDecrement() == 1 && !this.isShutdown.get()) {
                this.log.info(DATABASE_AVAILABLE_MSG, requirement.description(), this.databaseName);
                this.listeners.notify(AvailabilityListener::available);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void shutdown() {
        AtomicInteger atomicInteger = this.requirementCount;
        synchronized (atomicInteger) {
            if (this.isShutdown.getAndSet(true)) {
                return;
            }
            if (this.requirementCount.get() == 0) {
                this.listeners.notify(AvailabilityListener::unavailable);
            }
        }
    }

    @Override
    public boolean isAvailable() {
        return this.availability() == Availability.AVAILABLE;
    }

    @Override
    public boolean isShutdown() {
        return this.availability() == Availability.SHUTDOWN;
    }

    @Override
    public boolean isAvailable(long millis) {
        return this.availability(millis) == Availability.AVAILABLE;
    }

    @Override
    public void checkAvailable() throws UnavailableException {
        this.await(0L);
    }

    @Override
    public void await(long millis) throws UnavailableException {
        Availability availability = this.availability(millis);
        if (availability == Availability.AVAILABLE) {
            return;
        }
        String description = availability == Availability.UNAVAILABLE ? "Timeout waiting for database to become available and allow new transactions. Waited " + Format.duration(millis) + ". " + this.describeWhoIsBlocking() : "Database not available because it's shutting down";
        throw new UnavailableException(description);
    }

    private Availability availability() {
        if (this.isShutdown.get()) {
            return Availability.SHUTDOWN;
        }
        int count2 = this.requirementCount.get();
        if (count2 == 0) {
            return Availability.AVAILABLE;
        }
        assert (count2 > 0);
        return Availability.UNAVAILABLE;
    }

    private Availability availability(long millis) {
        Availability availability = this.availability();
        if (availability != Availability.UNAVAILABLE) {
            return availability;
        }
        long timeout = this.clock.millis() + millis;
        do {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                break;
            }
        } while ((availability = this.availability()) == Availability.UNAVAILABLE && this.clock.millis() < timeout);
        return availability;
    }

    @Override
    public void addListener(AvailabilityListener listener) {
        this.listeners.add(listener);
    }

    @Override
    public void removeListener(AvailabilityListener listener) {
        this.listeners.remove(listener);
    }

    public String describeWhoIsBlocking() {
        if (this.blockingRequirements.size() > 0 || this.requirementCount.get() > 0) {
            String causes = Iterables.join(", ", Iterables.map(AvailabilityRequirement::description, this.blockingRequirements));
            return this.requirementCount.get() + " reasons for blocking: " + causes + ".";
        }
        return "No blocking components";
    }

    private static class LoggingAvailabilityListener
    implements AvailabilityListener {
        private final Log log;
        private final String databaseName;

        LoggingAvailabilityListener(Log log, String databaseName) {
            this.log = log;
            this.databaseName = databaseName;
        }

        @Override
        public void available() {
            this.log.info("Database %s is ready.", this.databaseName);
        }

        @Override
        public void unavailable() {
            this.log.info("Database %s is unavailable.", this.databaseName);
        }
    }

    private static enum Availability {
        AVAILABLE,
        UNAVAILABLE,
        SHUTDOWN;

    }
}

